﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using UnityEditor;
using UnityEngine;
using System.IO.Compression;
using System.Threading.Tasks;

/// <summary>
/// Manager for the SimpleITK integration.
/// Since SimpleITK is a native library that requires binaries to be built for your target platform,
///  SimpleITK will be disabled by default and can be enabled through this class.
/// The binaries will be downloaded automatically.
/// </summary>
public class SimpleITKManager
{
    private static string SimpleITKDefinition = "UVR_USE_SIMPLEITK";

    public static bool IsSITKEnabled()
    {
        HashSet<string> defines =
            new HashSet<string>(
                PlayerSettings.GetScriptingDefineSymbolsForGroup(BuildTargetGroup.Standalone).Split(';'));
        return defines.Contains(SimpleITKDefinition);
    }

    public static void EnableSITK(bool enable)
    {
        BuildTarget activeTarget = EditorUserBuildSettings.activeBuildTarget;
        BuildTargetGroup activeGroup = BuildPipeline.GetBuildTargetGroup(activeTarget);
        if (enable && activeGroup != BuildTargetGroup.Standalone
                   && !EditorUtility.DisplayDialog("Build target does not support SimpleITK.",
                       $"SimpleITK is only supported in standalone builds and editor, and will not work on your selected build target ({activeTarget}).\n"
                       + "Enable SimpleITK for standalone (Windows, Linux, MacOS) and editor?", "Yes", "No"))
        {
            return;
        }

        if (enable && !HasDownloadedBinaries())
        {
            EditorUtility.DisplayDialog("Missing SimpleITK binaries",
                "You need to download the SimpleITK binaries before you can enable SimpleITK.", "Ok");
            return;
        }

        // Enable the UVR_USE_SIMPLEITK preprocessor definition for standalone target
        List<BuildTargetGroup> buildTargetGroups = new List<BuildTargetGroup>() { BuildTargetGroup.Standalone };
        foreach (BuildTargetGroup group in buildTargetGroups)
        {
            List<string> defines = PlayerSettings.GetScriptingDefineSymbolsForGroup(group).Split(';').ToList();
            defines.Remove(SimpleITKDefinition);
            if (enable)
                defines.Add(SimpleITKDefinition);
            PlayerSettings.SetScriptingDefineSymbolsForGroup(group, string.Join(";", defines));
        }

        // Save project and recompile scripts
        AssetDatabase.SaveAssets();
        AssetDatabase.Refresh();
#if UNITY_2019_3_OR_NEWER
        UnityEditor.Compilation.CompilationPipeline.RequestScriptCompilation();
#endif
    }

    public static bool HasDownloadedBinaries()
    {
        string binDir = GetBinaryDirectoryPath();
        string managedLibrary = "SimpleITKCSharpManaged.dll"; //note that while the file name is the same, this file is different for each OS
        string nativeLibrary =
#if UNITY_EDITOR_WIN
                    "SimpleITKCSharpNative.dll";
#elif UNITY_EDITOR_LINUX
                    "libSimpleITKCSharpNative.so";
#else
                    "libSimpleITKCSharpNative.dylib";
#endif  
        
        return Directory.Exists(binDir)
            && Directory.GetFiles(binDir).Length > 0
            && File.Exists(Path.Combine(binDir, managedLibrary))
            && File.Exists(Path.Combine(binDir, nativeLibrary));
    }

    public static async void DownloadBinaries()
    {
        try
        {
            string extractDirPath = GetBinaryDirectoryPath();
            string zipPath = Path.Combine(Directory.GetParent(extractDirPath).FullName, "SimpleITK.zip");
            if (HasDownloadedBinaries())
            {
                if (!EditorUtility.DisplayDialog("Download SimpleITK binaries",
                        "SimpleITK has already been downloaded. Do you want to delete it and download again?", "Yes", "Cancel"))
                {
                    return;
                }
            }

            EditorUtility.DisplayProgressBar("Downloading SimpleITK", "Downloading SimpleITK binaries.", 0);
                        
            if (!File.Exists(zipPath))
            {
                Directory.CreateDirectory(Path.GetDirectoryName(zipPath));
            }

            // Download binaries zip
            using (var client = new WebClient())
            {
                string downloadURL =
#if UNITY_EDITOR_WIN
                    "https://github.com/SimpleITK/SimpleITK/releases/download/v2.2.0/SimpleITK-2.2.0-CSharp-win64-x64.zip";
#elif UNITY_EDITOR_LINUX
                    "https://github.com/SimpleITK/SimpleITK/releases/download/v2.2.0/SimpleITK-2.2.0-CSharp-linux.zip";
#else
                    "https://github.com/SimpleITK/SimpleITK/releases/download/v2.2.0/SimpleITK-2.2.0-CSharp-macosx-10.9-anycpu.zip";
#endif          
                client.DownloadProgressChanged += (sender, eventArgs) => EditorUtility.DisplayProgressBar("Downloading SimpleITK", "Downloading SimpleITK binaries.", eventArgs.ProgressPercentage / 100f);
                await client.DownloadFileTaskAsync(downloadURL, zipPath);

                EditorUtility.DisplayProgressBar("Extracting SimpleITK", "Extracting SimpleITK.", 0.0f);

                if (!File.Exists(zipPath))
                {
                    Debug.Log(zipPath);
                    EditorUtility.DisplayDialog("Error downloadig SimpleITK binaries.",
                        "Failed to download SimpleITK binaries. Please check your internet connection.", "Close");
                    Debug.Log(
                        $"Failed to download SimpleITK binaries. You can also try to manually download from {downloadURL} and extract it to some folder inside the Assets folder.");
                    return;
                }

                try
                {
                    await ExtractZip(zipPath, extractDirPath);
                }
                catch (Exception ex)
                {
                    string errorString = $"Extracting binaries failed with error: {ex.Message}\n"
                                         + $"Please try downloading the zip from: {downloadURL}\nAnd extract it somewhere in the Assets folder.\n\n"
                                         + "The download URL can be copied from the error log (console).";
                    Debug.LogError(ex.ToString());
                    Debug.LogError(errorString);
                    EditorUtility.DisplayDialog("Failed to extract binaries.", errorString, "Close");
                }
            }

            File.Delete(zipPath);

            Debug.Log($"Simple ITK successfully downloaded.");
        }
        finally
        {
            EditorUtility.ClearProgressBar();
        }
    }

    private static async Task ExtractZip(string zipPath, string extractDirPath)
    {
        int filesUnzipped = 0;
        long bytesExtracted = 0;
        
        using (FileStream zipStream = new FileStream(zipPath, FileMode.Open))
        {
            using (ZipArchive archive = new ZipArchive(zipStream, ZipArchiveMode.Update))
            {
                if (!Directory.Exists(extractDirPath))
                    Directory.CreateDirectory(extractDirPath);

                foreach (ZipArchiveEntry entry in archive.Entries)
                {
                    if (entry.Name == "" || entry.Name.EndsWith("/")) continue;
                    filesUnzipped++;

                    string destFilePath = Path.Combine(extractDirPath, entry.Name);
                    Stream inStream = entry.Open();

                    using (Stream outStream = File.OpenWrite(destFilePath))
                    {
                        bytesExtracted += inStream.Length;
                        EditorUtility.DisplayProgressBar("Extracting SimpleITK", $"Extracting {filesUnzipped} files ({bytesExtracted / 1000_000}MB)", (float)filesUnzipped / archive.Entries.Count);
                        await inStream.CopyToAsync(outStream);
                    }
                }
            }
        }
        EditorUtility.DisplayProgressBar("Extracting SimpleITK", "Extracted SimpleITK.", 1.0f);
    }

    private static string GetBinaryDirectoryPath()
    {
        string dataPath = Application.dataPath;
        foreach (string file in Directory.EnumerateFiles(Application.dataPath, "*.*", SearchOption.AllDirectories))
        {
            // Search for magic file stored in Assets directory.
            // This is necessary for cases where the UVR plugin is stored in a subfolder (that's the case for the asset store version)
            if (Path.GetFileName(file) == "DONOTREMOVE-PathSearchFile.txt")
            {
                dataPath = Path.GetDirectoryName(file);
            }
        }

        return Path.Combine(dataPath, "3rdparty", "SimpleITK"); // TODO: What is UVR is in a subfolder?
    }
}