using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using UnityEngine;

#if UNITY_EDITOR
using UnityEditor;
#endif

//[ExecuteAlways]
[RequireComponent(typeof(VolumeObject))]
public class TransferFunction : MonoBehaviour
{
    // UnityVolumeRendering
    [Serializable]
    public struct TFControlPoint<T>
    {
        public float DensityValue;
        public T DataValue;
        
        public TFControlPoint (float densityValue, T dataValue)
        {
            this.DensityValue = densityValue;
            DataValue = dataValue;
        }
    }

    [SerializeField] private List<TFControlPoint<Color>> colorControlPoints = new();
    [SerializeField] private List<TFControlPoint<float>> alphaControlPoints = new();

    public Color[] lut;
    [SerializeField] private float windowLeft = 0;
    [SerializeField] private float windowWidth = 1;
    [SerializeField] private int lutSampleCount = 1000;

    [SerializeField] private bool needUpdate = true;

    private Texture2D uiTexture;

    private ComputeBuffer dataBuffer;
    private VolumeObject volume;

    private void OnEnable()
    {
        volume = GetComponent<VolumeObject>();
        volume.TransferFunction = this;
        needUpdate = true;
    }

    private void OnDisable()
    {
        if (dataBuffer != null)
        {
            dataBuffer.Release();
            dataBuffer = null;
        }

        volume.TransferFunction = null;
    }

    private void OnValidate()
    {
        needUpdate = true;
    }

    private Color[] GenerateLUT(int sampleCount, float minVal, float maxVal)
    {
        var resultLUT = new Color[sampleCount];
        var cols = new List<TFControlPoint<Color>>(colorControlPoints);
        var alphas = new List<TFControlPoint<float>>(alphaControlPoints);
        
        cols.Sort((a, b) => (a.DensityValue.CompareTo(b.DensityValue)));
        alphas.Sort((a, b) => (a.DensityValue.CompareTo(b.DensityValue)));
        
        // Add colour points at beginning and end
        if (cols.Count == 0 || cols[cols.Count - 1].DensityValue < 1.0f)
            cols.Add(new TFControlPoint<Color>(1.0f, Color.white));
        if (cols[0].DensityValue > 0.0f)
            cols.Insert(0, new TFControlPoint<Color>(0.0f, Color.white));

        // Add alpha points at beginning and end
        if (alphas.Count == 0 || alphas[alphas.Count - 1].DensityValue < 1.0f)
            alphas.Add(new TFControlPoint<float>(1.0f, 1.0f));
        if (alphas[0].DensityValue > 0.0f)
            alphas.Insert(0, new TFControlPoint<float>(0.0f, 0.0f));
        
        int numColours = cols.Count;
        int numAlphas = alphas.Count;
        int iCurrColour = 0;
        int iCurrAlpha = 0;

        for (int iX = 0; iX < sampleCount; iX++)
        {
            float t = iX / (float)(sampleCount - 1);
            // convert t to hounsfield units
            t = t * (maxVal - minVal) + minVal;
            t = (t - VolumeData.MIN_HOUNSFIELD) / (VolumeData.MAX_HOUNSFIELD - VolumeData.MIN_HOUNSFIELD);
            // 
            // map to provided values
            while (iCurrColour < numColours - 2 && cols[iCurrColour + 1].DensityValue < t)
                iCurrColour++;
            while (iCurrAlpha < numAlphas - 2 && alphas[iCurrAlpha + 1].DensityValue < t)
                iCurrAlpha++;

            var leftCol = cols[iCurrColour];
            var rightCol = cols[iCurrColour + 1];
            var leftAlpha = alphas[iCurrAlpha];
            var rightAlpha = alphas[iCurrAlpha + 1];

            var tCol = (Mathf.Clamp(t, leftCol.DensityValue, rightCol.DensityValue) - leftCol.DensityValue) / (rightCol.DensityValue - leftCol.DensityValue);
            var tAlpha = (Mathf.Clamp(t, leftAlpha.DensityValue, rightAlpha.DensityValue) - leftAlpha.DensityValue) / (rightAlpha.DensityValue - leftAlpha.DensityValue);

            tCol = Mathf.SmoothStep(0.0f, 1.0f, tCol);
            tAlpha = Mathf.SmoothStep(0.0f, 1.0f, tAlpha);

            var pixCol = rightCol.DataValue * tCol + leftCol.DataValue * (1.0f - tCol);
            pixCol.a = rightAlpha.DataValue * tAlpha + leftAlpha.DataValue * (1.0f - tAlpha);

            resultLUT[iX] = pixCol;
        }

        return resultLUT;
    }

    private void GenerateUITexture()
    {
        const int width = VolumeData.MAX_HOUNSFIELD - VolumeData.MIN_HOUNSFIELD;
        if (uiTexture == null)
        {
            uiTexture = new Texture2D(width, 1, TextureFormat.RGBAFloat, false);
        }
        var colors = GenerateLUT(width, VolumeData.MIN_HOUNSFIELD, VolumeData.MAX_HOUNSFIELD);
        uiTexture.SetPixels(colors);
        uiTexture.Apply();
    }

    public Texture2D GetUITexture()
    {
        if (uiTexture == null)
            GenerateUITexture();
        return uiTexture;
    }


    private void Update()
    {
        UploadData();
    }

    private void UploadData()
    {
        if (!needUpdate && dataBuffer != null) return;
        
        var volumeMinMax = volume.GetVolumeData().GetVolumeMinMax();
        lut = GenerateLUT(lutSampleCount, volumeMinMax.x, volumeMinMax.y);

        if (uiTexture != null)
        {
            GenerateUITexture();
        }
        
        dataBuffer ??= new ComputeBuffer(lut.Length, sizeof(float) * 4);

        if (lut.Length != dataBuffer.count)
        {
            dataBuffer.Release();
            dataBuffer = new ComputeBuffer(lut.Length, sizeof(float) * 4);
        }

        needUpdate = false;
        volume.OnHasChanged?.Invoke();

        //var data = ComputeLutCdf();
        
        dataBuffer.SetData(lut);
    }

    public bool IsMonotonicIncreasing()
    {
        if (alphaControlPoints.Count == 0) return true;
        var first = true;
        float lastAlpha = 0.0f;
        foreach (var tfControlPoint in alphaControlPoints.OrderBy(a => a.DensityValue))
        {
            if (first)
            {
                lastAlpha = tfControlPoint.DataValue;
                first = false;
                continue;
            }

            if (tfControlPoint.DataValue < lastAlpha)
            {
                return false;
            }

            lastAlpha = tfControlPoint.DataValue;
        }
        return true;
    }

    public void Bind(ComputeShader shader, ShaderIndices shaderIndices)
    {
        UploadData();

        if (IsMonotonicIncreasing())
        {
            shader.EnableKeyword("USE_DDA");
        } else
        {
            shader.DisableKeyword("USE_DDA");
        }
        
        shader.SetBuffer(shaderIndices.TraceKernel, shaderIndices.TransferFunctionBufferLut, dataBuffer);
        shader.SetInt(shaderIndices.TransferFunctionBufferSize, dataBuffer.count);
        shader.SetFloat(shaderIndices.TransferFunctionWindowLeft, windowLeft);
        shader.SetFloat(shaderIndices.TransferFunctionWindowWidth, windowWidth);
    }

    public TFControlPoint<Color> GetColorPoint(int index)
    {
        return colorControlPoints[index];
    }

    public void SetColorPoint(int index, TFControlPoint<Color> point)
    {
        colorControlPoints[index] = point;
        OnControlPointChange();
    }
    
    public void SetColorPoints(List<TFControlPoint<Color>> points)
    {
        colorControlPoints = points;
        OnControlPointChange();
    }

    public void AddColorPoint(TFControlPoint<Color> point)
    {
        colorControlPoints.Add(point);
        OnControlPointChange();
    }
    
    public void RemoveColorPoint(int index)
    {
        colorControlPoints.RemoveAt(index);
        OnControlPointChange();
    }

    public TFControlPoint<float> GetAlphaPoint(int index)
    {
        return alphaControlPoints[index];
    }
    
    public void SetAlphaPoint(int index, TFControlPoint<float> point)
    {
        alphaControlPoints[index] = point;
        OnControlPointChange();
    }

    public void SetAlphaPoints(List<TFControlPoint<float>> points)
    {
        alphaControlPoints = points;
        OnControlPointChange();
    }
    
    public void AddAlphaPoint(TFControlPoint<float> point)
    {
        alphaControlPoints.Add(point);
        OnControlPointChange();
    }

    public void RemoveAlphaPoint(int index)
    {
        alphaControlPoints.RemoveAt(index);
        OnControlPointChange();
    }

    private void OnControlPointChange()
    {
        needUpdate = true;
    #if UNITY_EDITOR
        EditorUtility.SetDirty(this);
        Update();
    #endif
    }

    public IEnumerable<TFControlPoint<Color>> ColorPointEnumerator()
    {
        return colorControlPoints;
    }

    public IEnumerable<TFControlPoint<float>> AlphaPointEnumerator()
    {
        return alphaControlPoints;
    }

    public void ClearControlPoints()
    {
        alphaControlPoints.Clear();
        colorControlPoints.Clear();
        alphaControlPoints.Add(new TFControlPoint<float>(0.2f, 0.0f));
        alphaControlPoints.Add(new TFControlPoint<float>(0.8f, 1.0f));
        colorControlPoints.Add(new TFControlPoint<Color>(0.5f, new Color(0.469f, 0.354f, 0.223f, 1.0f)));
        OnControlPointChange();
    }

    public VolumeObject GetVolume()
    {
        if (volume == null)
            volume = GetComponent<VolumeObject>();
        return volume;
    }

    // https://github.com/mlavik1/UnityVolumeRendering/blob/master/Assets/Scripts/TransferFunction/TransferFunctionDatabase.cs
    [Serializable]
    private struct SerializationData
    {
        public List<TFControlPoint<Color>> colorPoints;
        public List<TFControlPoint<float>> alphaPoints;
    }

    public void LoadFromFile(string filepath)
    {
        if(!File.Exists(filepath))
        {
            Debug.LogError(string.Format("File does not exist: {0}", filepath));
            return;
        }
        var jsonstring = File.ReadAllText(filepath);
        var data = JsonUtility.FromJson<SerializationData>(jsonstring);
        this.colorControlPoints = data.colorPoints;
        this.alphaControlPoints = data.alphaPoints;
        OnControlPointChange();
    }

    public void SaveToFile(string filepath)
    {
        var data = new SerializationData();
        data.colorPoints = this.colorControlPoints;
        data.alphaPoints = this.alphaControlPoints;
        string jsonstring = JsonUtility.ToJson(data);
        File.WriteAllText(filepath, jsonstring);
    }
}
