using System;
using Unity.Mathematics;
using UnityEngine;

[Serializable]
public class Environment
{
    private const uint DIMENSION = 512;
    private const uint SAMPLES = 64;

    [SerializeField] private VolumeLight volumeLight;
    [SerializeField] private ComputeShader setupShader;
    [SerializeField] private Texture2D environmentMap;
    [SerializeField] private Texture2D importanceMap;
    [SerializeField] private float strength = 1;

    private bool hasLight = false;
    
    public bool hasChanged = false;
    
    private float lastStrength;

    private void SetupEnv()
    {
        var _target = new RenderTexture((int)DIMENSION, (int)DIMENSION, 0, RenderTextureFormat.RFloat, RenderTextureReadWrite.Linear);
        _target.enableRandomWrite = true;
        _target.Create();
        
        var samples = (uint) math.sqrt(SAMPLES);
        setupShader.SetTexture(0, "_EnvMap", environmentMap);
        setupShader.SetTexture(0, "result", _target);
        setupShader.SetInts("output_size", (int) DIMENSION, (int) DIMENSION);
        setupShader.SetInts("output_size_samples", (int) (DIMENSION * samples), (int) (DIMENSION * samples));
        setupShader.SetInts("num_samples", (int) samples, (int) samples);
        setupShader.SetFloat("inv_samples", 1.0f / samples);
        setupShader.Dispatch(0, (int) DIMENSION, (int) DIMENSION, 1);
        
        // https://stackoverflow.com/questions/44264468/convert-rendertexture-to-texture2d
        importanceMap = new Texture2D((int) DIMENSION, (int) DIMENSION, TextureFormat.RFloat, true);
        var old_rt = RenderTexture.active;
        RenderTexture.active = _target;
        
        importanceMap.ReadPixels(new Rect(0, 0, importanceMap.width, importanceMap.height), 0, 0);
        importanceMap.Apply();

        RenderTexture.active = old_rt;
        lastStrength = strength;
        
        _target.Release();
    }

    public void Update()
    {
        if (!lastStrength.Equals(strength))
        {
            lastStrength = strength;
            hasChanged = true;
        }
        if (!hasLight && volumeLight != null)
        {
            hasLight = volumeLight.enabled;
            hasChanged = true;
            volumeLight.OnLightEnabled += () =>
            {
                hasLight = true;
            };
            volumeLight.OnLightDisabled += () =>
            {
                hasLight = false;
            };
            volumeLight.OnLightChanged += () =>
            {
                if (hasLight)
                {
                    hasChanged = true;
                }
            };
        }
    }

    public void Bind(ComputeShader shader, ShaderIndices shaderIndices)
    {
        if (environmentMap == null)
        {
            Debug.LogError("No environment set");
            return;
        }
        if (importanceMap == null)
        {
            SetupEnv();
        }

        if (hasLight)
        {
            volumeLight.Bind(shader, shaderIndices);
        }
        else
        {
            // todo not clean maybe fix otherwise
            shader.DisableKeyword("USE_LIGHT_SOURCE_POINT");
            shader.DisableKeyword("USE_LIGHT_SOURCE_DIRECTIONAL");
        }
        
        shader.SetTexture(shaderIndices.TraceKernel, shaderIndices.EnvironmentMap, environmentMap);
        shader.SetTexture(shaderIndices.TraceKernel, shaderIndices.EnvironmentImportanceMap, importanceMap);
        shader.SetFloat(shaderIndices.EnvironmentStrength, strength);
        shader.SetFloats(shaderIndices.EnvironmentImportanceInverseDimension, 1.0f / DIMENSION, 1.0f / DIMENSION);
        shader.SetInt(shaderIndices.EnvironmentImportanceBaseMip, math.floorlog2(DIMENSION));
    }
}
