using System;
using System.Collections.Generic;
using System.Linq;
using Unity.Mathematics;
using UnityEditor;
using UnityEngine;
using UnityEngine.Rendering;

// Heavy use of https://www.gamedeveloper.com/programming/gpu-ray-tracing-in-unity-part-1

[ImageEffectAllowedInSceneView]
[ExecuteInEditMode]
[RequireComponent(typeof(Camera))]
public class VolumetricPathTracing : MonoBehaviour
{
    [SerializeField] private ComputeShader pathTracingShader;
    [SerializeField] private ComputeShader tonemapBlitShader;

    [SerializeField] public Environment environment;

    // #if UNITY_EDITOR
    public SceneViewData sceneViewData; //add a button or bool or sth to copy over the values that you're editing in this component over to the sciptable object
    public bool ShowInSceneView = true;
    // #endif
    
    [SerializeField] public int maxSamples = 200;
    public CameraData cameraData = new CameraData();
    [SerializeField] private List<VolumeRenderState> States = new List<VolumeRenderState>();

    [SerializeField] private bool enableTonemapping = false;
    [SerializeField] private float gamma = 1.0f;
    [SerializeField] private float exposure = 1.0f;

    public bool blendWithScene = true;

    public RenderTexture outputTexture;
    public RenderTexture depthTexture;
    [SerializeField] private RenderTexture tempTexture;

    [SerializeField] private Camera Camera;
    
    private const string SceneCameraName = "SceneCamera";

    private void Start()
    {
        SetCameraData(ref cameraData, Camera);
        SetCameraData(ref sceneViewData.cameraData, sceneViewData.Camera);
        
        ResetGameViewTextures();
        ResetSceneViewTextures();
        
        ResetAllGameView();
        ResetAllSceneView();
    }

    private void OnEnable()
    {
        VolumeObject.OnVolumeAdded += RegisterObject;
        VolumeObject.OnVolumeRemoved += UnregisterObject;

        if (States != null)
        {
            foreach (VolumeRenderState volumeRenderState in States)
            {
                volumeRenderState.Destroy(false);
            }
            States.Clear();
        }
        
        if (sceneViewData != null && sceneViewData.States != null)
        {            
            foreach (VolumeRenderState volumeRenderState in sceneViewData.States)
            {
                if (volumeRenderState != null) volumeRenderState.Destroy(false);
            }
            sceneViewData.States.Clear();
        }

        foreach (VolumeObject volumeObject in VolumeObject.AllVolumes)
        {
            RegisterObject(volumeObject);
        }


        SetCameraData(ref cameraData, Camera);
        if(sceneViewData != null) SetCameraData(ref sceneViewData.cameraData, sceneViewData.Camera);
    }

    private void OnDisable()
    {
        VolumeObject.OnVolumeAdded -= RegisterObject;
        VolumeObject.OnVolumeRemoved -= UnregisterObject;

        foreach (VolumeRenderState volumeRenderState in States)
        {
            volumeRenderState.Destroy(false);
        }
        States.Clear();

        if (outputTexture != null)
        {
            outputTexture.Release();
        }
        if (depthTexture != null)
        {
            depthTexture.Release();
        }
        
        foreach (VolumeRenderState volumeRenderState in sceneViewData.States)
        {
            volumeRenderState.Destroy(false);
        }
        sceneViewData.States.Clear();

        if (sceneViewData.outputTexture != null)
        {
            sceneViewData.outputTexture.Release();
        }
        if (sceneViewData.depthTexture != null)
        {
            sceneViewData.depthTexture.Release();
        }
    }

#if UNITY_EDITOR
    private void OnValidate()
    {
        if (Camera.allCameras.Select(cam => cam.name).Contains(SceneCameraName))
            Debug.LogError($"Camera must not be named '{SceneCameraName}'! Otherwise shader cannot differentiate between scene view and game view cameras!", gameObject);
    }
#endif

    private void RegisterObject(VolumeObject volumeObject)
    {
        if (sceneViewData == null) return;

        bool notYetRegisteredHere = !States.Exists(volume => volume.VolumeObject != null && volume.VolumeObject.GetInstanceID() == volumeObject.GetInstanceID());
        if (notYetRegisteredHere)
        {
            States.Add(new VolumeRenderState(this, volumeObject, pathTracingShader, true));
            ResetGameViewTextures();
        }
        
        bool notYetRegisteredScene = !sceneViewData.States.Exists(volume => volume.VolumeObject != null && volume.VolumeObject.GetInstanceID() == volumeObject.GetInstanceID());
        if (notYetRegisteredScene)
        {
            sceneViewData.States.Add(new VolumeRenderState(this, volumeObject, pathTracingShader, false));
            ResetSceneViewTextures();
        }
    }

    private void UnregisterObject(VolumeObject volumeObject)
    {
        var index = States.FindIndex(volume => volume.VolumeObject != null && volume.VolumeObject.GetInstanceID() == volumeObject.GetInstanceID());
        if (index != -1)
        {
            States[index].Destroy();
            States.RemoveAt(index);
            ResetGameViewTextures();
        }

        var sceneIndex = sceneViewData.States.FindIndex(volume => volume.VolumeObject != null && volume.VolumeObject.GetInstanceID() == volumeObject.GetInstanceID());
        if (sceneIndex != -1)
        {
            sceneViewData.States[sceneIndex].Destroy();
            sceneViewData.States.RemoveAt(sceneIndex);
            ResetSceneViewTextures();
        }
    }

    void OnDestroy()
    {
        outputTexture = null;
    }
    
    public void ResetAllGameView()
    {
        foreach (VolumeRenderState volumeRenderState in States)
        {
            if (volumeRenderState.OutputIndex == -1)
            {
                volumeRenderState.Destroy(false);
                States.Remove(volumeRenderState);
            }
            volumeRenderState.NeedReset = true;
        }
    }
    public void ResetAllSceneView()
    {
        foreach (VolumeRenderState volumeRenderState in sceneViewData.States)
        {
            if (volumeRenderState.OutputIndex == -1)
            {
                volumeRenderState.Destroy(false);
                sceneViewData.States.Remove(volumeRenderState);
            }
            volumeRenderState.NeedReset = true;
        }
    }
    
    private void ResetGameViewTextures() //todo see if this can be merged with the below method
    {
        if (cameraData.dimensions.x == 0 || cameraData.dimensions.y == 0) return;

        ResetTexture(ref tempTexture, cameraData.dimensions, "TempTexture");

        if (States.Count <= 0) return;
        
        ResetTexture(ref outputTexture, cameraData.dimensions, "OutputTexture", States.Count);
        ResetTexture(ref depthTexture, cameraData.dimensions, "DepthTextures", States.Count, RenderTextureFormat.RFloat);

        int i = 0;
        foreach (VolumeRenderState renderState in States)
        {
            if (renderState.targetTexture == null)
            {
                renderState.OutputIndex = -1;
            }
            else
            {
                renderState.OutputIndex = i;
                i++;
                renderState.CopyTextureToMaster();
            }
        }
    }
    private void ResetSceneViewTextures() //todo see if this can be merged with the above method
    {
        if (sceneViewData.cameraData.dimensions.x == 0 || sceneViewData.cameraData.dimensions.y == 0) return;

        ResetTexture(ref sceneViewData.tempTexture, sceneViewData.cameraData.dimensions, "TempTexture");

        if (sceneViewData.States.Count <= 0) return;
        
        ResetTexture(ref sceneViewData.outputTexture, sceneViewData.cameraData.dimensions, "OutputTexture", sceneViewData.States.Count);
        ResetTexture(ref sceneViewData.depthTexture, sceneViewData.cameraData.dimensions, "DepthTextures", sceneViewData.States.Count, RenderTextureFormat.RFloat);

        int i = 0;
        foreach (VolumeRenderState renderState in sceneViewData.States)
        {
            if (renderState.targetTexture == null)
            {
                renderState.OutputIndex = -1;
            }
            else
            {
                renderState.OutputIndex = i;
                i++;
                renderState.CopyTextureToMaster();
            }
        }
    }

    public static void ResetTexture(ref RenderTexture texture, int2 dimensions, string name = "Texture", int? depth = null, RenderTextureFormat format = RenderTextureFormat.ARGBFloat)
    {
        if (texture != null)
        {
            texture.Release();
        }

        texture = new RenderTexture(dimensions.x, dimensions.y, 0, format, RenderTextureReadWrite.Linear);
        if (depth.HasValue)
        {
            texture.dimension = TextureDimension.Tex2DArray;
            texture.volumeDepth = depth.Value;
        }
        texture.enableRandomWrite = true;
        texture.name = $"{name}_{dimensions.x}x{dimensions.y}_frame{Time.renderedFrameCount}";
        texture.Create();
    }
    
    private bool SetCameraData(ref CameraData cameraData, Camera camera)
    {
        if (camera == null) return false;
        
        var hasChanged = false;

        if (!cameraData.fov.Equals(camera.fieldOfView))
        {
            cameraData.fov = camera.fieldOfView;
            hasChanged = true;
        }

        var dimensions = new int2(camera.scaledPixelWidth, camera.scaledPixelHeight);
        if (!cameraData.dimensions.Equals(dimensions))
        {
            cameraData.dimensions = dimensions;
            hasChanged = true;
        }

        var pos = camera.transform.position;
        if (!cameraData.position.Equals(pos))
        {
            cameraData.position = pos;
            hasChanged = true;
        }
        
        var camTransform = camera.cameraToWorldMatrix;
        if (!cameraData.cameraToWorld.Equals(camTransform))
        {
            cameraData.cameraToWorld = camera.cameraToWorldMatrix;
            hasChanged = true;
        }

        var camInvProj = camera.projectionMatrix.inverse;
        if (!cameraData.cameraInvProj.Equals(camInvProj))
        {
            cameraData.cameraInvProj = camInvProj;
            hasChanged = true;
        }

        return hasChanged;
    }

    private void OnRenderImage(RenderTexture source, RenderTexture destination)
    {
        if (Camera.current == null)
        {
            Graphics.Blit(source, destination);
            return;
        }

        if (Camera.current.name != SceneCameraName)
        {
            if (tempTexture == null || States.Count == 0)
            {
                Graphics.Blit(source, destination);
            }
            else if (!blendWithScene)
            {
                Graphics.Blit(outputTexture, destination, 0, 0);
            }
            else
            {
                var threadGroupsX = Mathf.CeilToInt(cameraData.dimensions.x / 8.0f);
                var threadGroupsY = Mathf.CeilToInt(cameraData.dimensions.y / 8.0f);
                tonemapBlitShader.SetTexture(0, "source", source);
                tonemapBlitShader.SetTexture(0, "destination", tempTexture);
                tonemapBlitShader.SetTexture(0, "volumes", outputTexture);
                tonemapBlitShader.SetTexture(0, "depth_textures", depthTexture);
                tonemapBlitShader.SetInt("volumeCount", States.Count);

#if UNITY_EDITOR
                if (States.Count > 64) Debug.LogWarning($"The maximum buffer size is set to 64, but {States.Count} volumes are being rendered. Try increasing MAX_NUMBER_OF_VOLUMES in TonemapBlit.compute or reducing the number of volumes!");
#endif

                if (enableTonemapping)
                {
                    tonemapBlitShader.EnableKeyword("ENABLE_TONEMAP");
                    tonemapBlitShader.SetFloat("gamma", gamma);
                    tonemapBlitShader.SetFloat("exposure", exposure);
                }
                else
                {
                    tonemapBlitShader.DisableKeyword("ENABLE_TONEMAP");
                }

                tonemapBlitShader.SetTextureFromGlobal(tonemapBlitShader.FindKernel("main"), "DepthTexture", "_CameraDepthTexture"); //todo cache or precompute main kernel

                tonemapBlitShader.Dispatch(tonemapBlitShader.FindKernel("main"), threadGroupsX, threadGroupsY, 1);

                Graphics.Blit(tempTexture, destination);
            }
        }
        else //in scene view
        {
            if (sceneViewData.tempTexture == null || sceneViewData.States.Count == 0)
            {
                Graphics.Blit(source, destination);
            }
            else if (!sceneViewData.blendWithScene)
            {
                Graphics.Blit(sceneViewData.outputTexture, destination, 0, 0);
            }
            else
            {
                var threadGroupsX = Mathf.CeilToInt(sceneViewData.cameraData.dimensions.x / 8.0f);
                var threadGroupsY = Mathf.CeilToInt(sceneViewData.cameraData.dimensions.y / 8.0f);
                tonemapBlitShader.SetTexture(0, "source", source);
                tonemapBlitShader.SetTexture(0, "destination", sceneViewData.tempTexture);
                tonemapBlitShader.SetTexture(0, "volumes", sceneViewData.outputTexture);
                tonemapBlitShader.SetTexture(0, "depth_textures", sceneViewData.depthTexture);
                tonemapBlitShader.SetInt("volumeCount", sceneViewData.States.Count);
#if UNITY_EDITOR
                if (sceneViewData.States.Count > 64) Debug.LogWarning($"The maximum buffer size is set to 64, but {sceneViewData.States.Count} volumes are being rendered. Try increasing MAX_NUMBER_OF_VOLUMES in TonemapBlit.compute or reducing the number of volumes!");
#endif

                if (sceneViewData.enableTonemapping)
                {
                    tonemapBlitShader.EnableKeyword("ENABLE_TONEMAP");
                    tonemapBlitShader.SetFloat("gamma", sceneViewData.gamma);
                    tonemapBlitShader.SetFloat("exposure", sceneViewData.exposure);
                }
                else
                {
                    tonemapBlitShader.DisableKeyword("ENABLE_TONEMAP");
                }

                tonemapBlitShader.SetTextureFromGlobal(tonemapBlitShader.FindKernel("main"), "DepthTexture", "_CameraDepthTexture"); //todo cache or precompute main kernel

                tonemapBlitShader.Dispatch(tonemapBlitShader.FindKernel("main"), threadGroupsX, threadGroupsY, 1);

                Graphics.Blit(sceneViewData.tempTexture, destination);
            }
        }
    }
    
    void OnPreRender()
    {
        Camera.depthTextureMode |= DepthTextureMode.Depth;
        if(sceneViewData.Camera != null) sceneViewData.Camera.depthTextureMode |= DepthTextureMode.Depth;
    }

    void OnPostRender()
    {
        environment.Update();

        if (SetCameraData(ref cameraData, Camera) || environment.hasChanged)
        {
            ResetAllGameView();
        }
        if (SetCameraData(ref sceneViewData.cameraData, sceneViewData.Camera) || environment.hasChanged)
        {
            ResetAllSceneView();
        }
        environment.hasChanged = false;

        if (outputTexture.width != cameraData.dimensions.x ||
            outputTexture.height != cameraData.dimensions.y)
        {
            ResetGameViewTextures();
        }
        
        if (sceneViewData.outputTexture.width != sceneViewData.cameraData.dimensions.x ||
            sceneViewData.outputTexture.height != sceneViewData.cameraData.dimensions.y)
        {
            ResetSceneViewTextures();
        }
    }
}