using System;
using System.Collections;
using Unity.Mathematics;
using UnityEngine;
using UnityEngine.Rendering;

[Serializable]
public class VolumeRenderState
{
    public int CurrentSample = 1;
    public readonly VolumeObject VolumeObject;
    public bool NeedReset = true;

    private int seed;

    private readonly ComputeShader computeShader;
    private readonly ShaderIndices shaderIndices;
    private readonly CommandBuffer commandBuffer;
    private readonly VolumetricPathTracing renderMaster;

    private PerfEval perfEval;
    [SerializeField] private float meanSampleTime = 0;
    [SerializeField] private float stdevSampleTime = 0;

    public RenderTexture targetTexture;
    public RenderTexture depthTexture;
    public int OutputIndex = 0;
    private GraphicsFence graphicsFence;
    private Coroutine renderRoutine;

    private bool isForGameView = true;

    public VolumeRenderState(VolumetricPathTracing renderMaster, VolumeObject volumeObject, ComputeShader computeShader, bool isForGameView)
    {
        VolumeObject = volumeObject;
        volumeObject.OnHasChanged += OnVolumeChanged;

        this.computeShader = computeShader;
        this.renderMaster = renderMaster;
        this.isForGameView = isForGameView;
        shaderIndices = new ShaderIndices(computeShader);
        commandBuffer = new CommandBuffer();
        seed = UnityEngine.Random.Range(int.MinValue, int.MaxValue);

        ResetTextures();

        renderRoutine = renderMaster.StartCoroutine(RenderLoop());
    }

    private IEnumerator RenderLoop()
    {
        while (true)
        {
            var startTime = Time.realtimeSinceStartup;
            yield return new WaitUntil(DispatchComputeSample);

            yield return new WaitUntil(() => graphicsFence.passed);

            // if (targetTexture.width != renderMaster.sceneViewData.outputTexture.width ||
            //     targetTexture.height != renderMaster.sceneViewData.outputTexture.height)
            // {
            //     continue;
            // }

            CopyTextureToMaster();

            var sampleTime = Time.realtimeSinceStartup - startTime;
            perfEval.AddSample(sampleTime);
            meanSampleTime = perfEval.Mean();
            stdevSampleTime = perfEval.Stdev();
            if((CurrentSample-10) % 10 <= 0) Debug.Log($"{seed}/{CurrentSample}: sample time {sampleTime*1000f}ms | avg {meanSampleTime*1000f}ms | stdev {stdevSampleTime*1000f}ms | avg fps {1f/meanSampleTime}");
        }
    }

    public void CopyTextureToMaster()
    {
        if (isForGameView)
        {
            if (targetTexture.width == renderMaster.outputTexture.width &&
               targetTexture.height == renderMaster.outputTexture.height)
            {
                Graphics.CopyTexture(targetTexture, 0, renderMaster.outputTexture, OutputIndex);
                Graphics.CopyTexture(depthTexture, 0, renderMaster.depthTexture, OutputIndex);
            }
        }
        else
        {
            if (renderMaster != null && renderMaster.sceneViewData != null &&
                targetTexture.width == renderMaster.sceneViewData.outputTexture.width &&
               targetTexture.height == renderMaster.sceneViewData.outputTexture.height)
            {
                Graphics.CopyTexture(targetTexture, 0, renderMaster.sceneViewData.outputTexture, OutputIndex);
                Graphics.CopyTexture(depthTexture, 0, renderMaster.sceneViewData.depthTexture, OutputIndex);
            }
        }
    }

    private bool DispatchComputeSample()
    {        
        var cameraDimensions = isForGameView ? renderMaster.cameraData.dimensions : renderMaster.sceneViewData.cameraData.dimensions;
        var maxSamples = isForGameView ? renderMaster.maxSamples : renderMaster.sceneViewData.maxSamples;
        
        if (!NeedReset && CurrentSample > maxSamples) return false;

      
        commandBuffer.Clear();
        commandBuffer.SetExecutionFlags(CommandBufferExecutionFlags.AsyncCompute);

        var threadGroupsX = Mathf.CeilToInt(cameraDimensions.x / 8.0f);
        var threadGroupsY = Mathf.CeilToInt(cameraDimensions.y / 8.0f);
        
        if (NeedReset)
        {
            NeedReset = false;
            perfEval = new PerfEval(maxSamples);
            if (targetTexture.width != cameraDimensions.x || targetTexture.height != cameraDimensions.y)
            {
                ResetTextures();
            }

            CurrentSample = 1;
            
            computeShader.SetTexture(shaderIndices.ClearResultKernel, shaderIndices.Result, targetTexture);
            computeShader.SetTexture(shaderIndices.ClearResultKernel, shaderIndices.DepthTexture, depthTexture);
            commandBuffer.DispatchCompute(computeShader, shaderIndices.ClearResultKernel, threadGroupsX, threadGroupsY, 1);
        }
        
        if (CurrentSample <= maxSamples)
        {
            computeShader.SetTexture(shaderIndices.TraceKernel, shaderIndices.Result, targetTexture);
            computeShader.SetTexture(shaderIndices.TraceKernel, shaderIndices.DepthTexture, depthTexture);

            if (isForGameView)
            {
                computeShader.SetMatrix(shaderIndices.CameraToWorld, renderMaster.cameraData.cameraToWorld);
                computeShader.SetMatrix(shaderIndices.CameraInvProj, renderMaster.cameraData.cameraInvProj);
                computeShader.SetInt(shaderIndices.ShowEnvironment, renderMaster.blendWithScene ? 0 : 1);
            }
            else
            {
                computeShader.SetMatrix(shaderIndices.CameraToWorld, renderMaster.sceneViewData.cameraData.cameraToWorld);
                computeShader.SetMatrix(shaderIndices.CameraInvProj, renderMaster.sceneViewData.cameraData.cameraInvProj);
                computeShader.SetInt(shaderIndices.ShowEnvironment, renderMaster.sceneViewData.blendWithScene ? 0 : 1);
            }

            computeShader.SetInt(shaderIndices.Seed, seed);
            computeShader.SetInt(shaderIndices.CurrentSample, CurrentSample++);

            VolumeObject.Bind(computeShader, shaderIndices);
            renderMaster.environment.Bind(computeShader, shaderIndices);

            commandBuffer.DispatchCompute(computeShader, shaderIndices.TraceKernel, threadGroupsX, threadGroupsY, 1);

            graphicsFence = commandBuffer.CreateGraphicsFence(GraphicsFenceType.CPUSynchronisation, SynchronisationStageFlags.AllGPUOperations);
            Graphics.ExecuteCommandBufferAsync(commandBuffer, ComputeQueueType.Background);

            if (CurrentSample > maxSamples) Debug.LogWarning($"{seed}: Last sample done. ({CurrentSample})");

            return true;
        }
        
        return false;
    }

    private void ResetTextures()
    {
        int2 dimensions = isForGameView ? renderMaster.cameraData.dimensions : renderMaster.sceneViewData.cameraData.dimensions;

        VolumetricPathTracing.ResetTexture(ref targetTexture, dimensions, VolumeObject.name);
        VolumetricPathTracing.ResetTexture(ref depthTexture, dimensions, $"Depth_{VolumeObject.name}", null, RenderTextureFormat.RFloat);
    }

    private void OnVolumeChanged()
    {
        NeedReset = true;
    }

    public void Destroy(bool stopRoutine = true)
    {
        if (stopRoutine)
        {
            renderMaster.StopCoroutine(renderRoutine);
        }
        if (targetTexture != null) targetTexture.Release();
        if (depthTexture != null) depthTexture.Release();
        if (commandBuffer != null) commandBuffer.Dispose();
    }
}