using UnityEngine;

/// <summary>
/// Use this class if your code is in this form:
/// <code>Update() => value *= 0.8f;</code>
/// and replace it with:
/// <code>value = FramerateIndependantExponentialInterpolation.Lerp(value, Vector3.zero, 13.3886f);</code>
/// where <paramref name="lambda"/> = ln(0.8f) * -60fps
/// <para>This is necessary to achieve the same motion on all framerates, which the first example doesn't. You can verify this by setting the framerate:</para>
/// <code>Application.targetFrameRate = 12;</code>
/// </summary>
public static class FramerateIndependantExponentialInterpolation
{
    public static float CalculateLambda(float t, float fps = 60) => -Mathf.Log(t) * fps;

    public static float Lerp(float a, float b, float lambda, float dt) => Mathf.Lerp(a, b, 1 - Mathf.Exp(-lambda * dt));
    public static float Lerp(float a, float b, float lambda) => Mathf.Lerp(a, b, 1 - Mathf.Exp(-lambda * Time.deltaTime));

    public static Vector2 Lerp(Vector2 a, Vector2 b, float lambda, float dt) => Vector2.Lerp(a, b, 1 - Mathf.Exp(-lambda * dt));
    public static Vector2 Lerp(Vector2 a, Vector2 b, float lambda) => Vector2.Lerp(a, b, 1 - Mathf.Exp(-lambda * Time.deltaTime));
    public static Vector3 Lerp(Vector3 a, Vector3 b, float lambda, float dt) => Vector3.Lerp(a, b, 1 - Mathf.Exp(-lambda * dt));
    public static Vector3 Lerp(Vector3 a, Vector3 b, float lambda) => Vector3.Lerp(a, b, 1 - Mathf.Exp(-lambda * Time.deltaTime));

    public static Quaternion Lerp(Quaternion a, Quaternion b, float lambda, float dt) => Quaternion.Lerp(a, b, 1 - Mathf.Exp(-lambda * dt));
    public static Quaternion Lerp(Quaternion a, Quaternion b, float lambda) => Quaternion.Lerp(a, b, 1 - Mathf.Exp(-lambda * Time.deltaTime));

    public static Vector3 Slerp(Vector3 a, Vector3 b, float lambda, float dt) => Vector3.Slerp(a, b, 1 - Mathf.Exp(-lambda * dt));
    public static Vector3 Slerp(Vector3 a, Vector3 b, float lambda) => Vector3.Slerp(a, b, 1 - Mathf.Exp(-lambda * Time.deltaTime));
    public static Quaternion Slerp(Quaternion a, Quaternion b, float lambda, float dt) => Quaternion.Slerp(a, b, 1 - Mathf.Exp(-lambda * dt));
    public static Quaternion Slerp(Quaternion a, Quaternion b, float lambda) => Quaternion.Slerp(a, b, 1 - Mathf.Exp(-lambda * Time.deltaTime));
}
