
using System.Reflection;
using UnityEditor;
using UnityEditor.ShortcutManagement;
using UnityEngine;

public static class NavigationShortcuts
{
    private const string To = "Navigate to Assets/";
    private const ShortcutModifiers control = ShortcutModifiers.Control;
    private const ShortcutModifiers shift = ShortcutModifiers.Shift;
    private const ShortcutModifiers alt = ShortcutModifiers.Alt;

    [Shortcut(To + "", KeyCode.Keypad0, control)] public static void NavigateToAssets() => ShowFolderContents("Assets");
    [Shortcut(To + "Art", KeyCode.Keypad1, control)] public static void NavigateToArt() => ShowFolderContents("Assets/Art");
    [Shortcut(To + "Audio", KeyCode.Keypad2, control)] public static void NavigateToAudio() => ShowFolderContents("Assets/Audio");
    [Shortcut(To + "Plugins", KeyCode.Keypad3, control)] public static void NavigateToPlugins() => ShowFolderContents("Assets/Plugins");
    [Shortcut(To + "Prefabs", KeyCode.Keypad4, control)] public static void NavigateToPrefabs() => ShowFolderContents("Assets/Prefabs");
    [Shortcut(To + "Scenes", KeyCode.Keypad5, control)] public static void NavigateToScenes() => ShowFolderContents("Assets/Scenes");
    [Shortcut(To + "Scripts", KeyCode.Keypad6, control)] public static void NavigateToScripts() => ShowFolderContents("Assets/Scripts");
    
    [Shortcut(To + "Art/Animations", KeyCode.Keypad1, control | shift)] public static void NavigateToAnimations() => ShowFolderContents("Assets/Art/Animations");
    [Shortcut(To + "Art/Fonts", KeyCode.Keypad2, control | shift)] public static void NavigateToFonts() => ShowFolderContents("Assets/Art/Fonts");
    [Shortcut(To + "Art/Images", KeyCode.Keypad3, control | shift)] public static void NavigateToImages() => ShowFolderContents("Assets/Art/Images");
    [Shortcut(To + "Art/Materials", KeyCode.Keypad4, control | shift)] public static void NavigateToMaterials() => ShowFolderContents("Assets/Art/Materials");
    [Shortcut(To + "Art/Models", KeyCode.Keypad5, control | shift)] public static void NavigateToModels() => ShowFolderContents("Assets/Art/Models");
    
    [Shortcut(To + "Scripts/Common", KeyCode.Keypad1, control | alt)] public static void NavigateToCommon() => ShowFolderContents("Assets/Scripts/Common");
    [Shortcut(To + "Scripts/Editor", KeyCode.Keypad2, control | alt)] public static void NavigateToEditor() => ShowFolderContents("Assets/Scripts/Editor");
    [Shortcut(To + "Scripts/Extensions", KeyCode.Keypad3, control | alt)] public static void NavigateToExtensions() => ShowFolderContents("Assets/Scripts/Extensions");
    
    
    [Shortcut("Window/General/Project", KeyCode.KeypadEnter, control)]
    private static void ShowProject() //This is what the source code does when you do Ctr+5
    {
        // EditorWindow.GetWindow<ProjectBrowser>(); //Reflection equivalent of this:
        
        System.Type projectBrowserType = System.Type.GetType("UnityEditor.ProjectBrowser,UnityEditor");

        MethodInfo getWindowMethod = typeof(EditorWindow)
            .GetMethod("GetWindow", BindingFlags.Public | BindingFlags.Static, null, System.Type.EmptyTypes, null);

        MethodInfo genericMethod = getWindowMethod.MakeGenericMethod(projectBrowserType);
        genericMethod.Invoke(null, null);
    }
    
    
    //Found at https://discussions.unity.com/t/tutorial-how-to-to-show-specific-folder-content-in-the-project-window-via-editor-scripting/685248/5
    
    /// <summary>
    /// Selects a folder in the project window and shows its content.
    /// Opens a new project window, if none is open yet.
    /// </summary>
    /// <param name="folderInstanceID">The instance of the folder asset to open.</param>
    public static void ShowFolderContents(string path)
    {
        var getInstanceIDMethod = typeof(AssetDatabase).GetMethod("GetMainAssetInstanceID",
            BindingFlags.Static | BindingFlags.NonPublic);
        int instanceID = (int)getInstanceIDMethod.Invoke(null, new object[] { path });
        ShowFolderContents(instanceID);
        
        ShowProject();
    }
    
    private static void ShowFolderContents(int folderInstanceID)
    {
        // Find the internal ProjectBrowser class in the editor assembly.
        Assembly editorAssembly = typeof(Editor).Assembly;
        System.Type projectBrowserType = editorAssembly.GetType("UnityEditor.ProjectBrowser");

        // This is the internal method, which performs the desired action.
        // Should only be called if the project window is in two column mode.
        MethodInfo showFolderContents = projectBrowserType.GetMethod(
            "ShowFolderContents", BindingFlags.Instance | BindingFlags.NonPublic);

        // Find any open project browser windows.
        Object[] projectBrowserInstances = Resources.FindObjectsOfTypeAll(projectBrowserType);

        if (projectBrowserInstances.Length > 0)
        {
            for (int i = 0; i < projectBrowserInstances.Length; i++)
                ShowFolderContentsInternal(projectBrowserInstances[i], showFolderContents, folderInstanceID);
        }
        else
        {
            EditorWindow projectBrowser = OpenNewProjectBrowser(projectBrowserType);
            ShowFolderContentsInternal(projectBrowser, showFolderContents, folderInstanceID);
        }
    }

    private static void ShowFolderContentsInternal(Object projectBrowser, MethodInfo showFolderContents, int folderInstanceID)
    {
        // Sadly, there is no method to check for the view mode.
        // We can use the serialized object to find the private property.
        SerializedObject serializedObject = new SerializedObject(projectBrowser);
        bool inTwoColumnMode = serializedObject.FindProperty("m_ViewMode").enumValueIndex == 1;

        if (!inTwoColumnMode)
        {
            // If the browser is not in two column mode, we must set it to show the folder contents.
            MethodInfo setTwoColumns = projectBrowser.GetType().GetMethod(
                "SetTwoColumns", BindingFlags.Instance | BindingFlags.NonPublic);
            setTwoColumns.Invoke(projectBrowser, null);
        }

        bool revealAndFrameInFolderTree = true;
        showFolderContents.Invoke(projectBrowser, new object[] { folderInstanceID, revealAndFrameInFolderTree });
    }

    private static EditorWindow OpenNewProjectBrowser(System.Type projectBrowserType)
    {
        EditorWindow projectBrowser = EditorWindow.GetWindow(projectBrowserType);
        projectBrowser.Show();

        // Unity does some special initialization logic, which we must call,
        // before we can use the ShowFolderContents method (else we get a NullReferenceException).
        MethodInfo init = projectBrowserType.GetMethod("Init", BindingFlags.Instance | BindingFlags.Public);
        init.Invoke(projectBrowser, null);

        return projectBrowser;
    }
}
